package ohd.hseb.hefs.utils.xml;

import ohd.hseb.hefs.utils.Variable;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import com.google.common.base.Objects;
import com.google.common.base.Throwables;

/**
 * Class for wrapping an object, giving it the {@link XMLReader} and {@link XMLWriter} interfaces.<br>
 * <br>
 * By default, the {@link #validate()} method will check the {@link #_required} flag against the value of {@link #get()}
 * and throw and exception if the variable is required but the value is null (unspecified).
 * 
 * @author alexander.garbarino
 */
public abstract class XMLVariable<T> extends Variable<T> implements XMLReader, XMLWriter, Cloneable
{
    private String _tag;

    private AttributeList _attributes = new AttributeList();

    /**
     * Defalts to false, so that XML elements are not required.
     */
    private boolean _required = false;

    protected XMLVariable(final String tag)
    {
        super();
        _tag = tag;
    }

    protected XMLVariable(final String tag, final T value)
    {
        super(value);
        _tag = tag;
    }

    public void setRequired(final boolean b)
    {
        _required = b;
    }

    /**
     * Wraps {@link AttributeList#addAttribute(String, Class, boolean)}.
     */
    @SuppressWarnings("unchecked")
    public XMLVariable addAttribute(final String name, final Class storageClass, final boolean required)
    {
        return _attributes.addAttribute(name, storageClass, required);
    }

    /**
     * Wraps {@link AttributeList#addAttribute(String, T, boolean)} passing in the storageObject's XML tag as the name.
     */
    @SuppressWarnings("hiding")
    //TODO I'm not sure what this is about!  Why is it hiding?
    public <T extends XMLVariable> void addAttribute(final T storageObject, final boolean required)
    {
        _attributes.addAttribute(storageObject.getXMLTagName(), storageObject, required);
    }

    /**
     * Wraps {@link AttributeList#addAttribute(String, T, boolean)}.
     */
    @SuppressWarnings("hiding")
    //TODO I'm not sure what this is about!  Why is it hiding?
    public <T extends XMLVariable> void addAttribute(final String name, final T storageObject, final boolean required)
    {
        _attributes.addAttribute(name, storageObject, required);
    }

    /**
     * Not sure if this is useful, since the {@link #addAttribute(String, Class, boolean)} returns the storage object.
     * This will access the storage object and return the value.
     * 
     * @param name Name of attribute.
     * @return Return the value from the attribute's underlying storage object.
     */
    public <V> V getAttributeValue(final String name)
    {
        @SuppressWarnings("unchecked")
        final Attribute<V> attr = _attributes.retrieve(name);
        if(attr != null)
        {
            return attr.getStorageObject().get();
        }
        return null;
    }

    /**
     * Sets storage object's contained value for the attribute.
     * 
     * @param name Name of the attribute.
     * @param o The value to set it to.
     */
    @SuppressWarnings("unchecked")
    public void setAttributeValue(final String name, final Object o)
    {
        @SuppressWarnings("rawtypes")
        final Attribute attr = _attributes.retrieve(name);
        if(attr != null)
        {
            attr.getStorageObject().set(o);
        }
    }

    public AttributeList getAttributeList()
    {
        return this._attributes;
    }

    @Override
    public XMLReader getReader()
    {
        return this;
    }

    @Override
    public XMLWriter getWriter()
    {
        return this;
    }

    /**
     * Defaults to {@link Object#toString()} for the value.
     */
    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        if(get() == null)
        {
            return null;
        }

        final Element element = XMLTools.createTextNodeElement(request, getXMLTagName(), writeVariable(get()));
        for(final Attribute attr: this._attributes)
        {
            attr.write(element);
        }
        return element;
    }

    /**
     * The contents of the xml node representing this variable. Defaults to {@code value}.{@link #toString()}.
     * 
     * @param value the current value of this variable
     * @return the text in the xml node to write
     */
    protected String writeVariable(final T value)
    {
//        if(value == null) //This is for testing only!!!! Need to remove eventually!
//        {
//            return "NULL";
//        }
        return value.toString();
    }

    /**
     * Reads the xml node contents and returns the new value of the variable. This does NOT change the value of this
     * variable! You must call {@link #set(Object)} or {@link #setValueOfElement(String, String)} to do that.
     * 
     * @param text the contents of the xml node
     * @return the new value
     */
    protected abstract T readVariable(String text) throws XMLReaderException;

    public void setXMLTagName(final String tag)
    {
        _tag = tag;
    }

    @Override
    public String getXMLTagName()
    {
        return _tag;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        if(Objects.equal(_tag, elementName))
        {
            set(readVariable(value));
        }
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
    {
        if(elementName.equals(getXMLTagName()))
        {
            _attributes.clearValues();
            _attributes.read(attr);
            _attributes.checkIfAllRequiredSet();
            _attributes.finalizeReading(); //We could call this in finalizeReading below.
        }
        return null;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public void validate() throws XMLReaderException
    {
        if((_required) && (get() == null))
        {
            throw new XMLReaderException("XML element " + _tag + " is required but was not specified.");
        }
    }

    /**
     * Creates a factory which just returns {@link #clone()}s of this object.
     */
    public XMLReaderFactory<? extends XMLVariable<T>> makeCloneFactory()
    {
        return new XMLReaderFactory<XMLVariable<T>>()
        {
            @Override
            public XMLVariable<T> get()
            {
                return XMLVariable.this.clone();
            }
        };
    }

    @SuppressWarnings("unchecked")
    @Override
    public XMLVariable<T> clone()
    {
        try
        {
            final XMLVariable<T> results = (XMLVariable<T>)super.clone();
            results._attributes = _attributes.clone();
            return results;
        }
        catch(final CloneNotSupportedException e)
        {
            throw Throwables.propagate(e);
        }
    }

    @Override
    public String toString()
    {
        return String.format("<xml %s:%s>", _tag, get());
    }

    @Override
    public boolean equals(final Object other)
    {
        if(this == other)
        {
            return true;
        }

        if(!(other instanceof XMLVariable<?>))
        {
            return false;
        }

        final XMLVariable<?> that = (XMLVariable<?>)other;
        return Objects.equal(this._tag, that._tag) && Objects.equal(this.get(), that.get())
            && (Objects.equal(this._attributes, that._attributes));
    }

    @Override
    public int hashCode()
    {
        return Objects.hashCode(get(), _tag);
    }
}
