package ohd.hseb.hefs.utils.xml;

import javax.xml.stream.XMLStreamWriter;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.google.common.base.Strings;

/**
 * An adapter class for creating {@link XMLWriter}s. This stores a tag name as well as a list of attributes. Note that
 * the {@link #getWriter()} method should not be overridden: all {@link XMLWriterAdapter} subclasses should return this
 * as the {@link XMLWriter}, otherwise the xml writing tools will get confused.
 * 
 * @author alexander.garbarino
 */
public abstract class XMLWriterAdapter implements XMLWriter
{
    protected String _tagName;

    private final AttributeList _attributes = new AttributeList();

    public XMLWriterAdapter(String tag)
    {
        _tagName = tag;
    }

    /**
     * Wraps {@link AttributeList#addAttribute(String, Class, boolean)}.
     */
    public XMLVariable addAttribute(String name, Class<? extends XMLVariable> storageClass, boolean required)
    {
        return _attributes.addAttribute(name, storageClass, required);
    }

    /**
     * Wraps {@link AttributeList#addAttribute(String, T, boolean)}.
     */
    public <T extends XMLVariable> void addAttribute(String name, T storageObject, boolean required)
    {
        _attributes.addAttribute(name, storageObject, required);
    }

    /**
     * Wraps {@link AttributeList#addAttribute(String, T, boolean)}, passing in the storageObject
     * {@link XMLVariable#getXMLTagName()} as the attribute name.
     */
    public <T extends XMLVariable> void addAttribute(T storageObject, boolean required)
    {
        _attributes.addAttribute(storageObject.getXMLTagName(), storageObject, required);
    }

    /**
     * Copies the attribute list elements from a {@link XMLReaderAdapter} to allow for both to use the same storage
     * objects.
     * 
     * @param writer
     */
    public void copyAttributesFrom(XMLReaderAdapter reader)
    {
        _attributes.clear();
        _attributes.addAll(reader.getAttributes());
    }

    //TODO protected
    public AttributeList getAttributes()
    {
        return _attributes;
    }

    protected void addAllAttributes(AttributeList list)
    {
        _attributes.addAll(list);
    }

    public boolean hasAttributes()
    {
        return !_attributes.isEmpty();
    }

    /**
     * @param writer The {@link XMLStreamWriter} to which to write the attributes.
     * @throws Exception
     */
    public void writeAttributes(XMLStreamWriter writer) throws Exception
    {
        if(hasAttributes())
        {
            _attributes.write(writer);
        }
    }

    @Override
    public String getXMLTagName()
    {
        return _tagName;
    }

    @Override
    public Element writePropertyToXMLElement(Document request) throws XMLWriterException
    {
        if(Strings.isNullOrEmpty(getXMLTagName()))
        {
            throw new IllegalStateException("Tag name is not specified.");
        }
        Element mainElement = request.createElement(this.getXMLTagName());
        _attributes.write(mainElement);
        return mainElement;
    }

    /**
     * DO NOT OVERRIDE THIS IN SUBCLASSES!
     */
    @Override
    public XMLWriter getWriter()
    {
        return this;
    }
}
