package ohd.hseb.hefs.utils.xml.vars;

import java.io.File;
import java.util.regex.Matcher;

import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLVariable;

/**
 * An {@link XMLVariable} that wraps a {@link File} object. The flag {@link #_checkForExistenceAndReadability} can be
 * set in the constructor instructing this whether or not to check for file existence and readability before setting the
 * file.<br>
 * <br>
 * Note that the value when written is the {@link File#toString()}, or {@link File#getPath()}, which is the exact string
 * that was provided in the constructor to File.
 * 
 * @author hank.herr
 */
public class XMLFile extends XMLVariable<File>
{
    public static final String DEFAULT_TAG = "file";

    private boolean _checkForExistenceAndReadability = false;

    public XMLFile(final boolean checkForExistenceAndReadability)
    {
        super(DEFAULT_TAG);
        _checkForExistenceAndReadability = checkForExistenceAndReadability;
    }

    public XMLFile(final String tag, final boolean checkForExistenceAndReadability)
    {
        super(tag);
        _checkForExistenceAndReadability = checkForExistenceAndReadability;
    }

    public XMLFile(final String tag, final File value, final boolean checkForExistenceAndReadability)
    {
        super(tag, value);
        _checkForExistenceAndReadability = checkForExistenceAndReadability;
    }

    @Override
    protected File readVariable(final String text) throws XMLReaderException
    {
        if(text == null)
        {
            return null;
        }
        final File file = new File(text);
        return file;
    }

    @Override
    protected String writeVariable(final File value)
    {
        //Ensure Linux file name separators are always used in output XML.
        return super.writeVariable(value).replaceAll(Matcher.quoteReplacement("\\"), "/");
    }

    @Override
    public void validate() throws XMLReaderException
    {
        super.validate(); //check required status satisfied
        if(get() != null)
        {
            if(_checkForExistenceAndReadability && (!get().exists() || !get().canRead()))
            {
                throw new XMLReaderException("Either the file, " + get().getAbsolutePath()
                    + ", does not exist or cannot be read.");
            }
        }
    }

}
