package ohd.hseb.hefs.utils.xml.vars;

import java.util.Calendar;
import java.util.Date;

import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLVariable;
import ohd.hseb.hefs.utils.xml.XMLWriterException;
import ohd.hseb.util.misc.HCalendar;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * An {@link XMLVariable} specifying a fixed date that uses a {@link Long} storage object. The date is stored using a
 * standard FEWS/OHD date element, which has tags date and time to specify the date as a String and assuming GMT.
 * 
 * @author hank.herr
 */
public class XMLFixedDate extends XMLVariable<Long>
{
    public static final String DEFAULT_TAG = "date";

    public XMLFixedDate()
    {
        super(DEFAULT_TAG);
    }

    public XMLFixedDate(final String tag)
    {
        super(tag);
        setupAttributes();
    }

    public XMLFixedDate(final Date value)
    {
        super(DEFAULT_TAG, value.getTime());
        setupAttributes();
    }

    public XMLFixedDate(final String tag, final Date value)
    {
        super(tag, value.getTime());
        setupAttributes();
    }

    private void setupAttributes()
    {
        this.addAttribute("date", XMLString.class, true);
        this.addAttribute("time", XMLString.class, true);
    }

    /**
     * @return True if the date has been set, false if the return of {@link #get()} is null.
     */
    private boolean prepareAttributes()
    {
        if(this.get() == null)
        {
            return false;
        }

        final Long dateInMillis = this.get();
        final String dateStr = HCalendar.buildDateStr(dateInMillis, "CCYY-MM-DD");
        final String timeStr = HCalendar.buildDateStr(dateInMillis, "hh:mm:ss");
        setAttributeValue("date", dateStr);
        setAttributeValue("time", timeStr);
        return true;
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        if(prepareAttributes())
        {
            return super.writePropertyToXMLElement(request);
        }
        return null;
    }

    @Override
    protected String writeVariable(final Long value)
    {
        //Writing is all in the attributes.
        return "";
    }

    @Override
    protected Long readVariable(final String text) throws XMLReaderException
    {
        //Reading is via the attributes, which should be set by the time this is called.
        final String dateStr = getAttributeValue("date");
        final String timeStr = getAttributeValue("time");

        //I will not use the HEFSDateTools here because I'm going to demand one format!
        final Calendar cal = HCalendar.convertStringToCalendar(dateStr + " " + timeStr, "CCYY-MM-DD hh:mm:ss");
        if(cal == null)
        {
            throw new XMLReaderException("The format for attribute date '" + dateStr + "' and/or time '" + timeStr
                + "' is not valid.");
        }
        return cal.getTimeInMillis();
    }

    @Override
    public void set(final Long value, final Object source)
    {
        super.set(value, source);
    }

}
