package ohd.hseb.hefs.utils.xml.vars;

import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLVariable;

/**
 * Super class used for XMLFloat, XMLDouble, and XMLInteger that provides bounds for the data and a method for checking
 * those bounds.
 * 
 * @author hank.herr
 * @param <T>
 */
public abstract class XMLNumber<T extends Number> extends XMLVariable<T>
{
    public static final String DEFAULT_TAG = "number";

    private T _lb = null;
    private T _ub = null;

    public XMLNumber(final String tag)
    {
        super(tag);
    }

    public XMLNumber(final T value)
    {
        super(DEFAULT_TAG, value);
    }

    public XMLNumber(final String tag, final T value)
    {
        super(tag, value);
    }

    /**
     * The bounds are inclusive!
     * 
     * @param lb
     * @param ub
     */
    public void setBounds(final T lb, final T ub)
    {
        if((lb != null) && (ub != null))
        {
            if(ub.doubleValue() < lb.doubleValue())
            {
                throw new IllegalArgumentException("Upper bound, " + ub + ", cannot be less than lower bound, " + lb
                    + ".");
            }
        }
        _lb = lb;
        _ub = ub;
    }

    public T getLowerBound()
    {
        return _lb;
    }

    public T getUpperBound()
    {
        return _ub;
    }

    public boolean isLowerBoundSet()
    {
        return _lb != null;
    }

    public boolean isUpperBoundSet()
    {
        return _ub != null;
    }

    public boolean areBoundsSet()
    {
        return (_lb != null) || (_ub != null);
    }

    /**
     * Checks the given number against the bounds.
     * 
     * @param num Null is allowed and will result in no checking.
     * @throws XMLReaderException If the bounds are exceeded.
     */
    public void checkBounds(final T num) throws XMLReaderException
    {
        if(num == null)
        {
            return;
        }
        if(!areBoundsSet())
        {
            return;
        }
        if((_lb != null) && (num.doubleValue() < _lb.doubleValue()))
        {
            throw new XMLReaderException(this.getXMLTagName() + " value, " + num
                + ", is smaller than the allowed lower bound, " + _lb + ".");
        }
        if((_ub != null) && (num.doubleValue() > _ub.doubleValue()))
        {
            throw new XMLReaderException(this.getXMLTagName() + " value, " + num
                + ", is larger than the allowed upper bound, " + _ub + ".");
        }
    }

    @Override
    public void set(final T value, final Object source)
    {
        try
        {
            checkBounds(value);
        }
        catch(final XMLReaderException e)
        {
            throw new IllegalArgumentException(e.getMessage());
        }
        super.set(value, source);
    }

}
