package ohd.hseb.hefs.utils.xml.vars;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import nl.wldelft.util.timeseries.ComplexEquidistantTimeStep;
import nl.wldelft.util.timeseries.SimpleEquidistantTimeStep;
import nl.wldelft.util.timeseries.TimeStep;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLVariable;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.HString;

import org.jfree.chart.axis.DateTickUnit;

import com.google.common.collect.Lists;

/**
 * Records a time step via two attributes: an {@link XMLInteger} multiplier and an {@link XMLListChoice} unit. Several
 * methods are provided to compute a JFreeChart tick unit, a millis time step, a time step string, and a FEWS
 * {@link TimeStep} instance.
 * 
 * @author hank.herr
 */
public class XMLTimeStep extends XMLVariable<String>
{
    public static final String DEFAULT_TAG = "timeStep";
    public static final String[] UNIT_CHOICES = {"hour", "hours", "day", "days", "week", "weeks", "month", "months",
        "year", "years"};
    public static String PERIOD_UNIT = "period";

    /**
     * {@link Collection} of special units that contains on the entry for "period", which is the standard time step
     * special unit.
     */
    public static final Collection<String> PERIOD_SPECIAL_UNITS = Arrays.asList(PERIOD_UNIT);

    private final List<String> _specialUnits = Lists.newArrayList();
    private final XMLInteger _multiplier = new XMLInteger("multiplier");
    private final XMLListChoice _unit = new XMLListChoice("unit", UNIT_CHOICES);

    public XMLTimeStep(final Collection<String> specialUnits)
    {
        super(DEFAULT_TAG, "");
        if(specialUnits != null)
        {
            _specialUnits.addAll(specialUnits);
        }
        _unit.getChoices().addAll(_specialUnits);
        setupAttributes();
    }

    public XMLTimeStep(final String tag, final Collection<String> specialUnits)
    {
        super(tag, "");
        if(specialUnits != null)
        {
            _specialUnits.addAll(specialUnits);
        }
        _unit.getChoices().addAll(_specialUnits);
        setupAttributes();
    }

    /**
     * May throw an {@link IllegalArgumentException} if unit or multiplier is invalid.
     * 
     * @param tag
     * @param unit
     * @param multiplier
     * @param specialUnits Units that are treated as special, meaning that the quantity has no meaning.
     */
    public XMLTimeStep(final String tag, final String unit, final int multiplier, final Collection<String> specialUnits)
    {
        super(tag, "");
        if(specialUnits != null)
        {
            _specialUnits.addAll(specialUnits);
        }
        _unit.getChoices().addAll(_specialUnits);
        setupAttributes();
        _multiplier.set(multiplier);
        _unit.set(unit);
    }

    public int getMultiplier()
    {
        return _multiplier.get();
    }

    public void setMultiplier(final Integer multiplier)
    {
        _multiplier.set(multiplier);
    }

    public String getUnit()
    {
        return _unit.get();
    }

    public void setUnit(final String unit)
    {
        _unit.set(unit);
    }

    protected Collection<String> getSpecialUnits()
    {
        return _specialUnits;
    }

    public Collection<String> getUnitChoices()
    {
        return _unit.getChoices();
    }

    public boolean isUnitYear()
    {
        return getUnit().contains("year");
    }

    public boolean isUnitMonth()
    {
        return getUnit().contains("month");
    }

    public boolean isSpecialUnit()
    {
        return _specialUnits.contains(_unit.get());
    }

    /**
     * @return The interval in milliseconds if it is a regular interval; i.e., if it is not a year or month unit (which
     *         are irregular) and if it is not a special unit. In those cases, the value {@link Long#MIN_VALUE} is
     *         returned.
     */
    public long computeIntervalInMillis()
    {
        if(isSpecialUnit() || isUnitYear() || isUnitMonth())
        {
            return Long.MIN_VALUE;
        }
        return HCalendar.computeIntervalValueInMillis("" + _multiplier.get() + " " + _unit.get());
    }

    private void setupAttributes()
    {
        this.addAttribute("multiplier", _multiplier, true);
        this.addAttribute("unit", _unit, false);
        _unit.getChoices().addAll(_specialUnits);
        _multiplier.setBounds(0, null);
    }

    /**
     * Parses the provided string and stores the resules in {@link #_multiplier} and {@link #_unit}. This throws an
     * exception if the parsing fails. This works consistently with
     * {@link HCalendar#computeIntervalValueInMillis(String)}.
     * 
     * @param timeStepStr
     */
    public void parseTimeStepStr(final String timeStepStr) throws XMLReaderException
    {
        final int pos = HString.findFirstNonNumericCharacter(timeStepStr);
        String multStr = timeStepStr.substring(0, pos).trim();
        if(multStr.isEmpty())
        {
            multStr = "1";
        }

        if((pos < 0) || (pos >= timeStepStr.length()))
        {
            _multiplier.set(_multiplier.readVariable(multStr));
            _unit.set(null);
        }
        else
        {
            _multiplier.set(_multiplier.readVariable(multStr));
            _unit.set(_unit.readVariable(timeStepStr.substring(pos).trim()));
        }
    }

    /**
     * @return A String specifying the time step that the GraphGen aggregators and other stuff can use.
     */
    public String buildTimeStepStr()
    {
        try
        {
            return XMLTools.computeTimeStepStringFromComponents("" + _multiplier.get(), _unit.get(), _specialUnits);
        }
        catch(final XMLReaderException e)
        {
            throw new IllegalArgumentException("Invalid time step: " + _multiplier.get() + " " + _unit.get());
        }
    }

    public TimeStep computeFEWSTimeStep()
    {
        if(_specialUnits.contains(_unit.get()))
        {
            throw new IllegalArgumentException("Special time steps cannot be computed here.");
        }
        else if(_unit.equals("year"))
        {
            return ComplexEquidistantTimeStep.YEAR;
        }
        else if(_unit.equals("month"))
        {
            return ComplexEquidistantTimeStep.MONTH;
        }
        else
        {
            return SimpleEquidistantTimeStep.getInstance(computeIntervalInMillis());
        }
    }

    /**
     * @return A JFreeChart tick unit. Null is returned if the unit is period.
     */
    public DateTickUnit computeDateTickUnit()
    {
        if(_specialUnits.contains(_unit.get()))
        {
            return null;
        }

        DateTickUnit tickUnit = null;
        if(_unit.get().equals("year"))
        {
            tickUnit = new DateTickUnit(DateTickUnit.YEAR, _multiplier.get());
        }
        else if(_unit.get().equals("month"))
        {
            tickUnit = new DateTickUnit(DateTickUnit.MONTH, _multiplier.get());
        }
        else if(_unit.get().equals("week"))
        {
            tickUnit = new DateTickUnit(DateTickUnit.DAY, _multiplier.get() * 7);
        }
        else if(_unit.get().equals("day"))
        {
            tickUnit = new DateTickUnit(DateTickUnit.DAY, _multiplier.get());
        }
        else if(_unit.get().equals("hour"))
        {
            tickUnit = new DateTickUnit(DateTickUnit.HOUR, _multiplier.get());
        }
        return tickUnit;
    }

    @Override
    protected String writeVariable(final String value)
    {
        //Writing is all in the attributes.
        return "";
    }

    @Override
    protected String readVariable(final String text) throws XMLReaderException
    {
        //Reading is handled in the attributes
        return "";
    }

    /**
     * @return True if the given {@link XMLTimeStep} has a unit of {@link #PERIOD_UNIT}.
     */
    public static boolean isPeriod(final XMLTimeStep timeStep)
    {
        return timeStep.getUnit().equals(PERIOD_UNIT);
    }
}
